import CredentialsProvider from "next-auth/providers/credentials";
import { PrismaAdapter } from "@next-auth/prisma-adapter";
import prisma from "@/lib/prismaconfig";
import { getUserByEmail, getuserLogByEmpId, userLog } from '@/lib/services/userService';
import { comparePassword } from '@/lib/utils/authUtils';

import { getPrimaryMenuItems } from "@/lib/services/primaryMenuService";
import { getSecondaryMenuItems } from "@/lib/services/secondaryMenuService";
import { getTertiaryMenuItems } from "@/lib/services/tertiaryMenuService";
import { getuserPermissionById } from "@/lib/services/userpermissions";

const authOptions = {
  adapter: PrismaAdapter(prisma),

  providers: [
    CredentialsProvider({
      name: "Credentials",
      credentials: {
        email: { label: "Email", type: "text" },
        password: { label: "Password", type: "password" },
      },

      async authorize(credentials) {
        if (!credentials.email || !credentials.password)
          throw new Error("Email and Password are required.");

        const user = await getUserByEmail(credentials.email);
        if (!user) throw new Error("User not found.");

        const isMatch = await comparePassword(credentials.password, user.user_hash_password);
        if (!isMatch) throw new Error("Incorrect password.");

        // ✅ Log start time
        const now = new Date();
        const date = now.toISOString().split("T")[0];
        const time = now.toTimeString().split(" ")[0];

        const existingLog = await getuserLogByEmpId(user.user_pid, date);
        if (!existingLog) {
          await userLog({
            log_createdBy: user.user_pid,
            log_date: date,
            log_startTime: time,
            log_endTime: null,
            log_isdelete: "active",
          });
        }

        // ✅ Fetch menus
        const primaryMenus = await getPrimaryMenuItems();
        const secondaryMenus = await getSecondaryMenuItems();
        const tertiaryMenus = await getTertiaryMenuItems();

        // ✅ Fetch user permissions
        const userPermissions = await getuserPermissionById(user.user_role);

        return {
          user_pid: user.user_pid,
          user_empid: user.user_empid,
          user_name: user.user_name,
          user_email: user.user_email,
          user_role: user.user_role,
          user_role_name: user.role.role_name,
          user_profile: user.user_profile,

          primaryMenus,
          secondaryMenus,
          tertiaryMenus,
          userPermissions,
        };
      },
    }),
  ],

  session: { strategy: "jwt", maxAge: 3600 },

  callbacks: {
    async jwt({ token, user }) {
      if (user) {
        Object.assign(token, user);
      }
      return token;
    },

    async session({ session, token }) {
      session.user = {
        user_pid: token.user_pid,
        user_empid: token.user_empid,
        name: token.user_name,
        email: token.user_email,
        role: token.user_role,
        role_name: token.user_role_name,
        profile: token.user_profile,
        primaryMenus: token.primaryMenus || [],
        secondaryMenus: token.secondaryMenus || [],
        tertiaryMenus: token.tertiaryMenus || [],
        userPermissions: token.userPermissions || {},
      };
      return session;
    },
  },

  pages: { signIn: "/", signOut: "/" },
  secret: process.env.NEXTAUTH_SECRET,
};

export default authOptions;
